// $License: NOLICENSE
//--------------------------------------------------------------------------------

/**
  Содержит сценарий по управлению серввера ТехноДок

  @file $relPath
  @copyright $copyright
  @author SMS-Automation
*/

//--------------------------------------------------------------------------------
// Libraries used (#uses)
#uses "Technodoc/Core/technodocServer"
#uses "Technodoc/Core/technodocServerSettings"

//--------------------------------------------------------------------------------
// Variables and Constants

//--------------------------------------------------------------------------------
// Константы для мониторинга локального экземпляра ТехноДока с помощью скрипта technodocWatcher.ctl

// Значение пути до корневой папки установленного локально экземпляра ТехноДока
const string TECHNODOC_ROOT_FOLDER_PATH = getPath(BIN_REL_PATH) + "TechnoDoc";

// Название сервиса
const string TECHNODOC_SERVICE_NAME = _WIN32 ? "TechnoDocServer" : "technodoc-server";

// Максимальное время недоступности сервера, после которого будет выполнен перезапуск
const int MAX_TIME_IN_UNHEALTHY_STATE_LOCAL_SERVER_SEC = 60;

// Интервал опроса состояния сервера
const int MONITORING_INTERVAL_LOCAL_SERVER_SEC = 5;

// Задержка для прогрева сервиса, чтобы убедиться, что с ним все в порядке
const int DELAY_AFTER_START_SEREVICE_SEC = 30;

// Флаг отладки скрипта Technodoc.Server.ctl для вывода дополнительных сообщений.
// Чтобы включить, необходимо в менеджере сценариев добавить -dbg MonitorDebug. Пример: "Technodoc.Server.ctl -dbg MonitorDebug"
const string MONITORING_DEBUG_FLAG = "MonitorDebug";

//--------------------------------------------------------------------------------
/** Мониторинг и управление локальным экземпляром ТехноДока
*/
main()
{
  if(!isTechnodocServiceRunning(TECHNODOC_SERVICE_NAME))
  {
    DebugFTN(MONITORING_DEBUG_FLAG, "Сервис ТехноДока не запущен, попытка запуска сервиса.");
    if(!isTechnodocServiceExists(TECHNODOC_SERVICE_NAME) && !createTechnodocService(TECHNODOC_ROOT_FOLDER_PATH))
    {
      throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось создать службу ТехноДока. Обратитесь к администратору. Скрипт мониторинга завершит свою работу."));
      return;
    }

    if(!startTechnodocService(TECHNODOC_ROOT_FOLDER_PATH) || !isTechnodocServiceRunning(TECHNODOC_SERVICE_NAME))
    {
      throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось запустить службу ТехноДока. Обратитесь к администратору. Скрипт мониторинга завершит свою работу."));
      return;
    }

    delay(DELAY_AFTER_START_SEREVICE_SEC);

    if(!isTechnodocServiceRunning(TECHNODOC_SERVICE_NAME))
    {
      throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось запустить службу ТехноДока. Обратитесь к администратору. Скрипт мониторинга завершит свою работу."));
      return;
    }
  }

  time lastHealthyServerStateTime = getCurrentTime();
  while(true)
  {
      delay(MONITORING_INTERVAL_LOCAL_SERVER_SEC);

      // Если сервер доступен, то ничего не делаем
      if(isTechnodocServerAlive(TECHNODOC_LOCAL_HTTP_ADDRESS))
      {
        DebugFTN(MONITORING_DEBUG_FLAG, "Текущий сервер с адресом \"" + TECHNODOC_LOCAL_HTTP_ADDRESS + "\" доступен");
        lastHealthyServerStateTime = getCurrentTime();
        continue;
      }

      // Если неработоспособность сервера не превысила максимальный таймаут, то ничего не делаем
      int secondsInUnhealthyState = (int)(getCurrentTime() - lastHealthyServerStateTime);
      if(secondsInUnhealthyState < MAX_TIME_IN_UNHEALTHY_STATE_LOCAL_SERVER_SEC)
      {
        DebugFTN(MONITORING_DEBUG_FLAG, "Текущий сервер с адресом \"" + TECHNODOC_LOCAL_HTTP_ADDRESS + "\" недоступен \"" + secondsInUnhealthyState + "\" секунд. Максимальное время недоступности, после которого будет выполнен перезапуск сервера \"" + MAX_TIME_IN_UNHEALTHY_STATE_LOCAL_SERVER_SEC + "\" секунд.");
        continue;
      }

      throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Время недоступности сервера ТехноДока превысило максимальное время. Будет произведена попытка перезапуска."));

      if(isTechnodocServiceRunning(TECHNODOC_SERVICE_NAME))
      {
        DebugFTN(MONITORING_DEBUG_FLAG, "Сервис ТехноДока запущен, но не отвечает. Попытка выполнить перезапуск сервиса.");
        stopTechnodocService(TECHNODOC_ROOT_FOLDER_PATH);
        startTechnodocService(TECHNODOC_ROOT_FOLDER_PATH);
      }
      else
      {
        DebugFTN(MONITORING_DEBUG_FLAG, "Сервис ТехноДока не запущен. Попытка выполнить запуск сервиса.");
        startTechnodocService(TECHNODOC_ROOT_FOLDER_PATH);
      }
  }
}

//--------------------------------------------------------------------------------
//@private members
//--------------------------------------------------------------------------------

/** Запущен ли сервис ТехноДока
  @param serviceName Имя сервиса ТехноДока
  @return Возвращает true если сервис ТехноДока запущен, иначе false
*/
private bool isTechnodocServiceRunning(string serviceName)
{
  DebugFTN(MONITORING_DEBUG_FLAG, "Начата проверка состояния запуска сервиса ТехноДока");

  int returnCode = -1;
  string out, err;
  string command = "";
  if(_WIN32)
  {
    command = "sc query \"" + serviceName + "\" | find \"RUNNING\"";
    returnCode = system(command, out, err);
  }
  else if(_UNIX)
  {
    command = "systemctl is-active --quiet \"" + serviceName + "\"";
    returnCode = system(command, out, err);
  }
  else
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось проверить состояние сервиса ТехноДока. Неподдерживаемая операционная система."));
    return false;
  }

  DebugFTN(MONITORING_DEBUG_FLAG, "Выполнена команда проверки существования сервиса ТехноДока: \"" + command + "\"");

  if(returnCode != 0)
  {
    throwError(makeError("", PRIO_WARNING, ERR_CONTROL, 0, __FUNCTION__ , "Сервис ТехноДока не запущен: " + out + " " + err));
    return false;
  }

  DebugFTN(MONITORING_DEBUG_FLAG, "Сервис ТехноДока запущен. Результат выполнения запуска: " + out + " " + err);

  return true;
}

/** Запустить сервис ТехноДока
  @param technodocRootFolderPath Корневая папка ТехноДока
  @return Возвращает true если удалось запустить сервис ТехноДока, иначе false
*/
private bool startTechnodocService(string technodocRootFolderPath)
{
  DebugFTN(MONITORING_DEBUG_FLAG, "Начата попытка запуска сервиса ТехноДока");

  string scriptName = "";

  if(_WIN32)
  {
    scriptName = "startTechnoDocServerSrv.bat";
  }
  else if(_UNIX)
  {
    scriptName = "startTechnoDocServerDaemon.sh";
  }
  else
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось выполнить запуск сервиса ТехноДока. Неподдерживаемая операционная система."));
    return false;
  }

  string scriptPath = technodocRootFolderPath + "/scripts/" + scriptName;

  if(!isfile(scriptPath))
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось найти скрипт запуска службы ТехноДока: \"" + scriptPath + "\". Обратитесь к администратору."));
    return false;
  }

  string out, err;
  int returnCode = system(scriptPath, out, err);

  if(returnCode != 0)
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Скрипт запуска службы ТехноДока завершился с ошибками: " + out + " " + err + ". Обратитесь к администратору."));
    return false;
  }

  DebugFTN(MONITORING_DEBUG_FLAG, "Запуск сервиса ТехноДока выполнен успешно");

  return true;
}

/** Остановить сервис ТехноДока
  @param technodocRootFolderPath Корневая папка ТехноДока
  @return Возвращает true если удалось остановить сервис ТехноДока, иначе false
*/
private bool stopTechnodocService(string technodocRootFolderPath)
{
  DebugFTN(MONITORING_DEBUG_FLAG, "Начата попытка остановки сервиса ТехноДока");

  string scriptName = "";

  if(_WIN32)
  {
    scriptName = "stopTechnoDocServerSrv.bat";
  }
  else if(_UNIX)
  {
    scriptName = "stopTechnoDocServerDaemon.sh";
  }
  else
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось выполнить остановку сервиса ТехноДока. Неподдерживаемая операционная система."));
    return false;
  }

  string scriptPath = technodocRootFolderPath + "/scripts/" + scriptName;

  if(!isfile(scriptPath))
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось найти скрипт остановки службы ТехноДока: \"" + scriptPath + "\". Обратитесь к администратору."));
    return false;
  }

  string out, err;
  int returnCode = system(scriptPath, out, err);

  if(returnCode != 0)
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Скрипт остановки службы ТехноДока завершился с ошибками: " + out + " " + err + ". Обратитесь к администратору."));
    return false;
  }

  DebugFTN(MONITORING_DEBUG_FLAG, "Остановка сервиса ТехноДока выполнена успешно");

  return true;
}

/** Существет ли сервис ТехноДока
  @param serviceName Имя сервиса ТехноДока
  @return Возвращает true если сервис ТехноДока существует, иначе false
*/
private bool isTechnodocServiceExists(string serviceName)
{
  DebugFTN(MONITORING_DEBUG_FLAG, "Начата проверка существования сервиса ТехноДока");

  int returnCode = -1;
  string out, err;
  if(_WIN32)
  {
    returnCode = system("sc query \"" + serviceName + "\"", out, err);
  }
  else if(_UNIX)
  {
    returnCode = system("systemctl cat \"" + serviceName + "\"", out, err);
  }
  else
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось выполнить проверку существования сервиса ТехноДока. Неподдерживаемая операционная система."));
    return false;
  }

  if(returnCode != 0)
  {
    throwError(makeError("", PRIO_WARNING, ERR_CONTROL, 0, __FUNCTION__ , "Выполнение проверки существования сервиса ТехноДока завершилось с ошибками: " + out + " " + err + "."));
    return false;
  }

  DebugFTN(MONITORING_DEBUG_FLAG, "Сервис ТехноДока существует");

  return true;
}

/** Создать сервис ТехноДока
  @param technodocRootFolderPath Корневая папка ТехноДока
  @return Возвращает true если удалось создать сервис ТехноДока, иначе false
*/
private bool createTechnodocService(string technodocRootFolderPath)
{
  DebugFTN(MONITORING_DEBUG_FLAG, "Начата попытка создания сервиса ТехноДока");

  string scriptName = "";

  if(_WIN32)
  {
    scriptName = "createTechnoDocServerSrv.bat";
  }
  else if(_UNIX)
  {
    scriptName = "createTechnoDocServerDaemon.sh";
  }
  else
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось выполнить создание сервиса ТехноДока. Неподдерживаемая операционная система."));
    return false;
  }

  string scriptPath = technodocRootFolderPath + "/scripts/" + scriptName;

  if(!isfile(scriptPath))
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Не удалось найти скрипт создания службы ТехноДока: \"" + scriptPath + "\". Обратитесь к администратору."));
    return false;
  }

  string out, err;
  int returnCode = system(scriptPath, out, err);

  if(returnCode != 0)
  {
    throwError(makeError("", PRIO_SEVERE, ERR_CONTROL, 0, __FUNCTION__ , "Скрипт создания службы ТехноДока авершился с ошибками: " + out + " " + err + ". Обратитесь к администратору."));
    return false;
  }

  DebugFTN(MONITORING_DEBUG_FLAG, "Создание сервиса ТехноДока выполнено успешно");

  return true;
}

