var AuthenticationService = (function () {
    function AuthenticationService(location, $injector, $q, notificationService, language) {
        this.location = location;
        this.$injector = $injector;
        this.$q = $q;
        this.notificationService = notificationService;
        this.language = language;
        this.userStorageKey = 'user';
        this.accessTokenStorageKey = 'accessToken';
        this.messagePortEventName = 'message';
    }
    AuthenticationService.prototype.logout = function () {
        var _this = this;
        var authenticationProvider = this.$injector.get('authenticationProvider');
        authenticationProvider.signOut()
            .catch(function (error) {
            _this.notificationService.errorMessage(_this.language.getElementValue('failedLoadDataServerMsg'), error);
        }).finally(function () {
            localStorage.removeItem(_this.accessTokenStorageKey);
            localStorage.removeItem(_this.userStorageKey);
            _this.location.url('/login');
        });
    };
    ;
    AuthenticationService.prototype.isLoggedIn = function () {
        return _.isDefined(this.user);
    };
    AuthenticationService.prototype.getAccessTokenHeader = function () {
        var accessToken = localStorage.getItem(this.accessTokenStorageKey);
        return _.isDefined(accessToken)
            ? 'Bearer ' + accessToken
            : '';
    };
    AuthenticationService.prototype.refreshToken = function () {
        var _this = this;
        if (this.inflightAuthRequest) {
            return this.inflightAuthRequest;
        }
        var currentToken = localStorage.getItem(this.accessTokenStorageKey);
        if (!currentToken) {
            this.location.url('/login');
            return this.$q.reject();
        }
        var deferred = this.$q.defer();
        this.inflightAuthRequest = deferred.promise;
        var handler = function (event) {
            var data = event.data;
            if (data.type === 'done') {
                _this.setAccessToken(data.token);
                deferred.resolve(data.token);
            }
            else {
                deferred.reject(data.type === 'error' ? data.error : 'Failed refresh token');
                if (_this.isLoggedIn()) {
                    _this.logout();
                }
                else {
                    _this.location.url('/login');
                }
            }
            _this.worker.port.removeEventListener(_this.messagePortEventName, handler);
            _this.inflightAuthRequest = null;
        };
        var refreshMessageToToken = {
            type: 'refresh',
            accessToken: currentToken
        };
        if (this.worker) {
            this.worker.port.addEventListener(this.messagePortEventName, handler);
            this.worker.port.postMessage(refreshMessageToToken);
        }
        else {
            var coreConfiguration = this.$injector.get('coreConfigurationProvider');
            coreConfiguration.getVersion()
                .then(function (version) {
                _this.worker = new SharedWorker("ng1/dist/src/Core/Workers/refresh-token-worker.js?v=".concat(version));
                _this.worker.port.start();
                _this.worker.port.addEventListener(_this.messagePortEventName, handler);
                _this.worker.port.postMessage(refreshMessageToToken);
            })
                .catch(function (error) {
                deferred.reject(error);
            });
        }
        return deferred.promise;
    };
    AuthenticationService.prototype.updatePermissions = function () {
        var _this = this;
        var user = this.user;
        return this.$injector.get('userProvider').getUserPermissions(user.id)
            .then(function (permissions) {
            user.permissions = permissions;
            localStorage.setItem(_this.userStorageKey, JSON.stringify(user));
        });
    };
    AuthenticationService.prototype.hasPermission = function (permission) {
        var user = this.user;
        return _.isDefined(user)
            ? _.contains(user.permissions, permission)
            : false;
    };
    AuthenticationService.prototype.hasPermissions = function (permissions) {
        var user = this.user;
        return _.isDefined(user)
            ? _.some(permissions, function (permission) { return _.contains(user.permissions, permission); })
            : false;
    };
    AuthenticationService.prototype.hasPermissionStartWith = function (permissionPrefix) {
        var user = this.user;
        return _.isDefined(user)
            ? _.any(user.permissions, function (permission) { return permission.startsWith(permissionPrefix); })
            : false;
    };
    Object.defineProperty(AuthenticationService.prototype, "user", {
        get: function () {
            var userJson = localStorage.getItem(this.userStorageKey);
            return userJson
                ? JSON.parse(userJson)
                : null;
        },
        enumerable: false,
        configurable: true
    });
    AuthenticationService.prototype.getUser = function (accessToken) {
        var decodedToken = this.parseAccessToken(accessToken);
        var deferred = this.$q.defer();
        this.$injector.get('userProvider').getUserPermissions(decodedToken.Id).then(function (permissions) {
            var user = {
                id: decodedToken.Id,
                name: decodedToken.Name,
                permissions: permissions,
            };
            deferred.resolve(user);
        });
        return deferred.promise;
    };
    AuthenticationService.prototype.setAccessToken = function (accessToken) {
        var _this = this;
        localStorage.setItem(this.accessTokenStorageKey, accessToken);
        this.getUser(accessToken)
            .then(function (user) {
            localStorage.setItem(_this.userStorageKey, JSON.stringify(user));
        });
    };
    AuthenticationService.prototype.parseAccessToken = function (accessToken) {
        var payload = accessToken.split('.')[1];
        var base64 = payload.replace('-', '+').replace('_', '/');
        while (base64.length % 4) {
            base64 += '=';
        }
        return JSON.parse(atob(base64));
    };
    AuthenticationService.$inject = [
        '$location',
        '$injector',
        '$q',
        'notificationService',
        'language',
    ];
    return AuthenticationService;
}());
app.service('authentication', AuthenticationService);
