// $License: NOLICENSE
//--------------------------------------------------------------------------------
/**
  Функции по работе с получением значений тегов

  @file $relPath
  @copyright $copyright
  @author SMS-Automation
*/

//--------------------------------------------------------------------------------
// Libraries used (#uses)
#uses "Technodoc/Common/technodocTime"
#uses "Technodoc/Common/technodocDatapoint"


//--------------------------------------------------------------------------------
// Variables and Constants

const int PERIOD_RESULT_VALUE = 1;
const int PERIOD_RESULT_TYPE = 2;
const int PERIOD_RESULT_STATUS = 3;
const int PERIOD_RESULT_TIME = 4;

// Константа для обозначения отсутствия значения
const string TD_NULL_VALUE = "%null%";

// Константа, когда не опредлен тип переменной
const string TD_NO_VAR_TYPE = "0";

//--------------------------------------------------------------------------------
//@public members
//--------------------------------------------------------------------------------

/** Вернуть значение точки данных по имени функции
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @param functionName Имя функции для получения данных
  @return Значение точки данных в виде массива со структурой: значение, статус и время
*/
dyn_string getByFunctionName(string dp, time startTime, time endTime, string functionName)
{
  dyn_string result;
  string correctAddress = getCorrectDpAddress(dp);

  switch (strtolower(functionName))
  {
    case "getdatapointvalue":
      result = getDatapointValue(correctAddress);
      break;

    case "getdatapointvalueontime":
      result = getDatapointValueOnTime(correctAddress, startTime);
      break;

    case "getlastdatapointvalue":
      result = getLastDatapointValue(correctAddress, endTime);
      break;

    case "getlastdatapointvalueonperiod":
      result = getLastDatapointValueOnPeriod(correctAddress, startTime, endTime);
      break;

    case "getmaxdatapointvalue":
      result = getMaxDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getmindatapointvalue":
      result = getMinDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getaveragedatapointvalue":
      result = getAverageDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getmaxcontinuousdatapointvalue":
      result = getMaxContinuousDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getmincontinuousdatapointvalue":
      result = getMinContinuousDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getaveragecontinuousdatapointvalue":
      result = getAverageContinuousDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getsumdatapointvalue":
      result = getSumDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getcountdatapointvalue":
      result = getCountDatapointValue(correctAddress, startTime, endTime);
      break;

    case "getweightaveragedatapointvalue":
      result = getWeightAverageDatapointValue(correctAddress, startTime, endTime);
      break;
  }

  return result;
}

/** Вернуть значение пользовательского бита
  @param dp Адрес точки данных
  @param timestamp Временная метка
  @param bitNumber Номер бита
  @return Значение бита в виде массива со структурой: значение, статус и время
*/
dyn_string getUserbitValueOnTime(string dp, time timestamp, int bitNumber)
{
 dp = getFullDpAddress(dp);
 return dpGetAsyncEx(dp + ":_offline.._userbit" + bitNumber, timestamp);
}

/** Вернуть список значение точки данных за период
  @param dp Адрес точки данных
  @param startTime Дата начала периода
  @param endTime Дата окончания периода
  @return Массив массивов строк со значениями тега за промежуток времени
          Массивы строк содержат в себе [значение, статус, метка времени]
*/
dyn_dyn_string getDatapointValuesOnPeriod(string dp, time startTime, time endTime)
{
  string correctAddress = getCorrectDpAddress(dp);
  dyn_dyn_anytype values = getPeriodValues(correctAddress, startTime, endTime);

  dyn_dyn_mixed result;
  for (int i = 1; i <= dynlen(values); i++)
  {
    dynAppend(result,
      makeDynString(
        values[i][PERIOD_RESULT_VALUE],
        values[i][PERIOD_RESULT_TYPE],
        values[i][PERIOD_RESULT_STATUS],
        formatTime("%d.%m.%Y %H:%M:%S", values[i][PERIOD_RESULT_TIME], ".%d")));
  }

  return result;
}

//--------------------------------------------------------------------------------
//@private members
//--------------------------------------------------------------------------------

/** Вернуть текущее значение точки данных
  @param dp Адрес точки данных
  @return Значение точки данных в виде массива со структурой: значение, статус и время
*/
private dyn_string getDatapointValue(string dp)
{
  dyn_string result;
  string val = TD_NULL_VALUE, tim = (string)getCurrentTime(), type = TD_NO_VAR_TYPE;
  bool badStat = true;

  if (dp != "")
  {
    dpGet(dp, val);
    dpGet(dp + ":_offline.._stime", tim);
    dpGet(dp + ":_offline.._bad", badStat);
  }

  dynAppend(result, val);
  dynAppend(result, val == TD_NULL_VALUE ? TD_NO_VAR_TYPE : getType(val));
  dynAppend(result, convertBoolStatusToString(badStat));
  dynAppend(result, tim);
  return result;
}

/** Вернуть значение точки данных за метку времени
  @param dp Адрес точки данных
  @param timestamp Метка времени
  @return Значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getDatapointValueOnTime(string dp, time timestamp)
{
  return dpGetAsyncEx(dp, timestamp);
}

/** Вернуть последнее значение точки данных за метку времени
  @param dp Адрес точки данных
  @param endTime Метка времени
  @return Последнее значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getLastDatapointValue(string dp, time endTime)
{
  return dpGetAsyncEx(dp, addMillisecondsTD(endTime, -1));
}

/** Вернуть последнее значение точки данных за период
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Последнее значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getLastDatapointValueOnPeriod(string dp, time startTime, time endTime)
{
  dyn_string result;
  dyn_string value = getDatapointValueOnTime(dp, endTime);

  if ((time)value[4] < startTime)
  {
    dynAppend(result, TD_NULL_VALUE);
    dynAppend(result, value[2]);
    dynAppend(result, "");
    dynAppend(result, endTime);
    return result;
  }
  return value;
}

/** Вернуть максимальное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @param useInitialValue Включать ли первое значение за пределами интервала
  @return Максимальное значение в виде массива [значение, статус, метка времени]
*/private dyn_string getMaxDpValue(string dp, time startTime, time endTime, bool useInitialValue)
{
  dyn_string result;

  dyn_dyn_mixed values = getPeriodValues(dp, startTime, endTime);
  int count = dynlen(values);

  dyn_string firstValue = getDatapointValueOnTime(dp, startTime);
  if(useInitialValue && (startTime > (time)firstValue[PERIOD_RESULT_TIME]) && (count == 0 || startTime < (time)values[1][PERIOD_RESULT_TIME]))
  {
    useInitialValue = true;
    dynInsertAt(values, makeDynMixed(firstValue[PERIOD_RESULT_VALUE], firstValue[PERIOD_RESULT_TYPE], firstValue[PERIOD_RESULT_STATUS], firstValue[PERIOD_RESULT_TIME]), 1);
  }

  count = dynlen(values);
  if (count == 0)
  {
    dynAppend(result, TD_NULL_VALUE);
    dynAppend(result, firstValue[2]);
    dynAppend(result, "");
    dynAppend(result, endTime);
    return result;
  }

  result = values[1];
  for (int i = 1; i <= count; i++)
  {
    if ((float)values[i][PERIOD_RESULT_VALUE] > (float)result[PERIOD_RESULT_VALUE])
      result = values[i];
  }

  return result;
}

/** Вернуть максимальное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Максимальное значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getMaxDatapointValue(string dp, time startTime, time endTime)
{
  return getMaxDpValue(dp,startTime,endTime, false);
}

/** Вернуть максимальное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Максимальное значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getMaxContinuousDatapointValue(string dp, time startTime, time endTime)
{
  return getMaxDpValue(dp, startTime, endTime, true);
}

/** Вернуть минимальное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @param useInitialValue Включать ли первое значение за пределами интервала
  @return Минимальное значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getMinDpValue(string dp, time startTime, time endTime, bool useInitialValue)
{
  dyn_string result;

  dyn_dyn_mixed values = getPeriodValues(dp, startTime, endTime);
  int count = dynlen(values);

  dyn_string firstValue = getDatapointValueOnTime(dp, startTime);
  if(useInitialValue && (startTime > (time)firstValue[PERIOD_RESULT_TIME]) && (count == 0 || startTime < (time)values[1][PERIOD_RESULT_TIME]))
  {
    useInitialValue = true;
    dynInsertAt(values, makeDynMixed(firstValue[PERIOD_RESULT_VALUE], firstValue[PERIOD_RESULT_TYPE], firstValue[PERIOD_RESULT_STATUS], firstValue[PERIOD_RESULT_TIME]), 1);
  }

  count = dynlen(values);
  if (count == 0)
  {
    anytype val;
    dpGet(dp, val);
    dynAppend(result, TD_NULL_VALUE);
    dynAppend(result, getType(val));
    dynAppend(result, "");
    dynAppend(result, endTime);
    return result;
  }

  result = values[1];
  for (int i = 1; i <= count; i++)
  {
    if ((float)values[i][PERIOD_RESULT_VALUE] < (float)result[PERIOD_RESULT_VALUE])
      result = values[i];
  }

  return result;
}

/** Вернуть минимальное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Минимальное значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getMinDatapointValue(string dp, time startTime, time endTime)
{
  return getMinDpValue(dp, startTime, endTime, false);
}

/** Вернуть минимальное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Минимальное значение в виде массива [значение, статус, метка времени]
*/
private dyn_string getMinContinuousDatapointValue(string dp, time startTime, time endTime)
{
  return getMinDpValue(dp, startTime, endTime, true);
}

/** Вернуть среднее значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @param useInitialValue Включать ли первое значение за пределами интервала
  @return Среднее значение
*/
private float getAverageDpValue(string dp, time startTime, time endTime, bool useInitialValue)
{
  float sum = 0;

  dyn_dyn_mixed values = getPeriodValues(dp, startTime, endTime);
  int count = dynlen(values);

  dyn_string firstValue = getDatapointValueOnTime(dp, startTime);
  if(useInitialValue && (startTime > (time)firstValue[PERIOD_RESULT_TIME]) && (count == 0 || startTime < (time)values[1][PERIOD_RESULT_TIME]))
  {
    useInitialValue = true;
    sum += firstValue[PERIOD_RESULT_VALUE];
  }
  else
  {
    useInitialValue = false;
  }

  if (count == 0 && !useInitialValue)
    return 0;

  for (int i = 1; i <= count; i++)
  {
    sum = sum + values[i][PERIOD_RESULT_VALUE];
  }

  return sum / ( useInitialValue ? count+1 : count);
}

/** Вернуть среднее значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Среднее значение
*/
private float getAverageDatapointValue(string dp, time startTime, time endTime)
{
  return getAverageDpValue(dp, startTime, endTime, false);
}

/** Вернуть среднее значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Среднее значений
*/
private float getAverageContinuousDatapointValue(string dp, time startTime, time endTime)
{
  return getAverageDpValue(dp, startTime, endTime, true);
}

/** Вернуть сумму значений точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Сумма значений
*/
private float getSumDatapointValue(string dp, time startTime, time endTime)
{
  dyn_dyn_mixed values = getPeriodValues(dp, startTime, endTime);
  int count = dynlen(values);

  if (count == 0)
    return 0;

  float sum = 0;
  for (int i = 1; i <= count; i++)
  {
    sum = sum + values[i][PERIOD_RESULT_VALUE];
  }

  //DebugTN("xmlrpc|GetSumDatapointValue|" + dp + ": за период " + (string)startTime + " - " + (string)endTime + ": " + (string)count + " = " + (string)sum);

  return sum;
}

/** Вернуть количество значений точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Количество значений
*/
private int getCountDatapointValue(string dp, time startTime, time endTime)
{
  dyn_dyn_mixed values = getPeriodValues(dp, startTime, endTime);
  return dynlen(values);
}

/** Вернуть средневзвешенное значение точки данных за промежуток времени
  @param dp Адрес точки данных
  @param startTime Дата начала
  @param endTime Дата окончания
  @return Средневзвешенное значение
*/
private float getWeightAverageDatapointValue(string dp, time startTime, time endTime)
{
    dyn_dyn_mixed values = getPeriodValues(dp, startTime, endTime);
    int count = dynlen(values);

    float sum = 0;
    float weightsum = 0;

    bool useInitialValue = false;
    dyn_string firstValue = getDatapointValueOnTime(dp, startTime);
    if(startTime > (time)firstValue[PERIOD_RESULT_TIME] && (count == 0 || startTime < (time)values[1][PERIOD_RESULT_TIME]))
    {
      useInitialValue = true;
    }


    if (count == 0 && !useInitialValue)
        return 0;

    if(useInitialValue)
    {
      time startInterval = startTime;

      time endInterval = count > 0? values[1][PERIOD_RESULT_TIME] : endTime ;
      int weight = period(endInterval - startInterval);
      sum = sum + (float)firstValue[PERIOD_RESULT_VALUE] * weight;
      weightsum = weightsum + weight;
          }

    for (int i = 1; i <= count; i++)
    {
        time startInterval = values[i][PERIOD_RESULT_TIME];
        if (startInterval < startTime)
            startInterval = startTime;

        time endInterval;
        if (i + 1 > count)
            endInterval = endTime;
        else
            endInterval = values[i + 1][PERIOD_RESULT_TIME];

        int weight = period(endInterval - startInterval);

        sum = sum + values[i][PERIOD_RESULT_VALUE] * weight;
        weightsum = weightsum + weight;
    }

    return sum / weightsum;
}

/** Вернуть значение точки данных на метку времени
  @param dp Адрес точки данных
  @param timestamp Временная метка
  @return Значение точки данных в виде массива [значение, статус, метка времени]
          Если у тега не задан архив, вернется текущее значение тега
*/
private dyn_string dpGetAsyncEx(string dp, time timestamp)
{
  dyn_string result;
  anytype val;
  string tim = "";
  bool badStatus = false;
  string start = timeToStringForUtcTD(addMillisecondsTD(timestamp, -1)), end = timeToStringForUtcTD(timestamp);
  dyn_string istDP = strsplit(dp, ":");

  string configToRead = "_offline.._value";
  //если 3 элемента в массиве, значит мы передали конфиг, который хотим прочитать
  if(dynlen(istDP) == 3)
  {
      configToRead = istDP[3];
      dp = istDP[1] + ":" + istDP[2];
  }

  dyn_dyn_anytype resultQuery;
  string sQuery = "SELECT '" + configToRead + "', '_offline.._stime', '_offline.._bad' FROM \'" + dp +
    "\' REMOTE \'" + istDP[1] + ":\' TIMERANGE(\"" + start + "\",\"" + end + "\",1,1)";

  dpQuery(sQuery,	resultQuery);

  /// Функция dpQuery с запросом SELECT ... TIMERANGE(startTs, endTs, modus, bonus) со значением bonus = 1
  /// должна возвращать все значения в интервале от startTs до endTs плюс как минимум одно значение до и после интервала.
  ///
  /// Но, начиная с какого-то времени из прошлого точки, функция перестает возвращать значения.
  /// Т.о., dpQuery не вернет значения в двух случаях: 1) у точки не задан архив, 2) архив есть, но выполняется описанный выше баг Каскад.

  bool hasArchiveValue = dynlen(resultQuery) > 1;

  if (hasArchiveValue)
  {
    for(int i = 2; i <= dynlen(resultQuery); i++) {

      time lastTime = (time)tim;
      if(timestamp >= resultQuery[i][3] && lastTime <= resultQuery[i][3]){
        val = resultQuery[i][2];
        tim = (string)resultQuery[i][3];
        badStatus = (bool)resultQuery[i][4];
      }
    }

    // не нашли данных с меткой времени <= timestamp
    if (tim == "")
    {
      dpGet(dp, val);
      tim = end;
      hasArchiveValue = false;
    }
  }
  else
  {
      dpGet(dp, val);
      tim = end;
  }

  dynAppend(result, hasArchiveValue ? val : TD_NULL_VALUE);
  dynAppend(result, getType(val));
  dynAppend(result, convertBoolStatusToString(badStatus));
  dynAppend(result, tim);
  return result;
}

/** Вернуть значение точки данных за период
  @param dp Адрес точки данных
  @param startTime Дата начала периода
  @param endTime Дата окончания периода
  @return Массив массивов объектов со значениями тега за промежуток времени
          Массивы строк содержат в себе [значение, статус, метка времени]
*/
private dyn_dyn_mixed getPeriodValues(string dp, time startTime, time endTime)
{
  dyn_dyn_mixed result;
  dyn_anytype values;
  dyn_time times;
  dyn_bool badStatuses;
  getPeriod(startTime, endTime, dp, values, times, badStatuses);

  for (int i = 1; i <= dynlen(values); i++)
  {
    if( times[i] < endTime )
    {
      dynAppend(
        result,
        makeDynMixed(values[i], getType(values[i]), convertBoolStatusToString(badStatuses[i]), times[i]));
    }
  }

  return result;
}

/** Вернуть значение точки данных за период
  @param startTime Дата начала периода
  @param endTime Дата окончания периода
  @param dp Адрес точки данных
  @param values Ссылка на список значений
  @param times Ссылка на список меток времени
  @param badStatuses Ссылка на список статусов
*/
private void getPeriod(time startTime, time endTime, string dp, dyn_mixed& values, dyn_string& times, dyn_bool& badStatuses)
{
  dp = getFullDpAddress(dp);
  int secInPart = 86400 / 2;
  float parts = (float)(endTime - startTime) / secInPart;
  int fullParts = (int)parts;
  float residueParts = parts - fullParts;
  dyn_dyn_anytype result;
  string start, end, sQuery;
  dyn_string istDP = strsplit(dp, ":");
  for (int d = 0; d < fullParts; d++)
  {
    start = timeToStringForUtcTD(startTime + (secInPart*d));
    end = timeToStringForUtcTD(startTime + (secInPart*d) + secInPart);

    //DebugTN(start,end);
    sQuery = "SELECT '_offline.._value', '_offline.._stime', '_offline.._bad' FROM \'" + dp +
      "\' REMOTE \'" + istDP[1] + ":\' TIMERANGE(\"" + start + "\",\"" + end + "\",1,0)";
    //DebugTN(sQuery);
    dpQuery(sQuery,	result);
    for(int i = 2; i <= dynlen(result); i++)
    {
       if(end == timeToStringForUtcTD(result[i][3]))continue;
       dynAppend(values, result[i][2]);
       dynAppend(times, (string)result[i][3]);
       dynAppend(badStatuses, (bool)result[i][4]);
    }
  }

  if (residueParts > 0) {
    start = timeToStringForUtcTD(endTime - (residueParts*secInPart));
    end = timeToStringForUtcTD(endTime);
    sQuery = "SELECT '_offline.._value', '_offline.._stime', '_offline.._bad' FROM \'" + dp +
      "\' REMOTE \'" + istDP[1] + ":\' TIMERANGE(\"" + start + "\",\"" + end + "\",1,0)";
    //DebugTN(sQuery);
    dpQuery(sQuery,	result);
    for(int i = 2; i <= dynlen(result); i++)
    {
       if(end == timeToStringForUtcTD(result[i][3]))continue;
       dynAppend(values, result[i][2]);
       dynAppend(times, (string)result[i][3]);
       dynAppend(badStatuses, (bool)result[i][4]);
    }
  }
}

/** Преобразовать статус из булевого значения в строку
  @param badStatus Статус значения
  @return Строковое представление статуса значения
*/
private string convertBoolStatusToString(bool badStatus)
{
  if (badStatus)
  {
    return "Bad";
  }
  else
  {
    return "Good";
  }
}
