// $License: NOLICENSE
//--------------------------------------------------------------------------------
/**
  Функции по работе с точками данных

  @file $relPath
  @copyright $copyright
  @author SMS-Automation
*/

//--------------------------------------------------------------------------------
// Libraries used (#uses)

//--------------------------------------------------------------------------------
// Variables and Constants


//--------------------------------------------------------------------------------
//@public members
//--------------------------------------------------------------------------------

/** Вернуть описание точки данных
  @param address Адрес точки данных
  @param dpGetDescriptionMode -
       0 - если описание не задано, возвращает полное имя элемента точки данных, т.е. DP.DPE1.DPE2.DPE3.[...].DPEN
      -1 - если описание не задано, возвращает только имя элемента точки данных или имя точки данных, т.е. для элемента точки данных DP.DPE1.DPE2 вернет DPE2, а для точки DP. вернет DP,
     ... - см. справку по функции dpGetDescription()
  @return Описание точки данных
*/
string getDatapointDescription(string address, int dpGetDescriptionMode = 0)
{
  string correctAddress = getCorrectDpAddress(address);
  return (string)dpGetDescription(correctAddress, dpGetDescriptionMode);
}

/** Вернуть единицу измерения точки данных
  @param address Адрес точки данных
  @return Единица измерения точки данных
*/
string getDatapointUnit(string address)
{
  string correctAddress = getCorrectDpAddress(address);
  return (string)dpGetUnit(correctAddress);
}

/** Формат измерения точки данных
  @param address Адрес точки данных
  @return Формат точки данных
*/
string getDatapointFormat(string address)
{
  string correctAddress = getCorrectDpAddress(address);
  return (string)dpGetFormat(correctAddress);
}

/** Вернуть короткий адрес точки данных без атрибута
  @param address Адрес точки данных
  @return Адрес точки данных без системы
*/
string getShortDatapointAddress(string address)
{
    dyn_string array = strsplit(address, ":");

    if(dynlen(array) > 2)
    {
          return array[1] + ":" + array[2];
    }
    else
    {
      return address;
    }
}


/** Вернуть полный адрес точки данных с именем системы
  @param address Адрес тега
  @return Полный адрес тега

  /// Адрес тега address имеет вид - [SystemName:]tagName[:config]
  /// Имя точки данных не должно совпадать с именем системы
  ///
  /// Пример: System1:dpName:_address, dpName:_corr.._value, System1:dpName
*/
string getCorrectDpAddress(string address)
{
   address = getFullDpAddress(address);
   dyn_string array = strsplit(address, ":");

   dyn_string names;
   dyn_uint ids;
   getSystemNames(names, ids);

   // добавляем имя системы в двух случаях:
   // 1) если в address нет символа ":"
   // 2) если в address есть один символ ":" и он используется для разделения имени точки с именем системы, а не для разделения имени точки с именем конфигурации
   //
   // при проверке условия 1) используется функция strpos, а не dynlen(array) == 1, из-за того, что функция strsplit(address, ":") может вернуть массив с одним элементом
   //                         либо если в address нет символа ":", либо в address один символ ":" и он является последним элементом строки address;
   if(address != "" && (strpos(address, ":") == -1 ||
                        (dynlen(array) == 2 && array[1] != "" && !dynContains(names, array[1]))))
   {
       // если имя системы не указано, используем имя собственной системы
       // (сделано по аналогии со стандартными функциями работы с точками данных dpGet, dpExists и т.д.)
       string correctDpAddress = getSystemName() + address;
       if (dpExists(correctDpAddress)) {
           return correctDpAddress;
       }
   }

   return address;
}

/** Получить полный адрес точки данных
  @param address Адрес тэга
  @return Полный адрес тега
*/
string getFullDpAddress(string address)
{
  dyn_string nameParts = strsplit(address, ".");

  // Проверяем сколько уровней у точки, уровни разделяются символом '.', если у точки 1 уровень, то добавляем к имени '.'
  // необходимо для корректного чтения данных для проектов с типом NGA (NextGenArchiver)
  if(dynlen(nameParts) == 1 && (strlen(address) > 1 && address[strlen(address) - 1] != "."
                                || strlen(address) == 1))
  {
    address = address + ".";
  }

  return address;
}

/** Вернуть результат проверки тега address в виде массива строк
  @param address Адрес точки данных
  @return Результат проверки тега address в виде массива строк, где
    [0] - текущее значение тега
    [1] - идентификатор статуса тега

    Список статусов тега:
    1 - нет архива
    2 - точка не создана
    3 - адрес не задан
    4 - точка создана, архив задан
*/
dyn_string checkDp(string address)
{
    dyn_string res;
    bool exists = dpExists(address);

    int status = 0;
    string value = "";

    if (exists)
    {
      // точка создана
      string shortAddress = getShortDatapointAddress(address);
      string archName;
      dpGet(shortAddress + ":_archive.1._class", archName);
      bool hasArchive = archName != "";

      dpGet(address, value);
      if (!hasArchive)
      {
        // нет архива
        status = 1;
      }
      else
      {
        // архив задан
        status = 4;
      }
    }
    else
    {
      // точка не создана
      status = 2;
    }
    dynAppend(res, value);
    dynAppend(res, (string)status);

    return res;
}

//--------------------------------------------------------------------------------
//@private members
//--------------------------------------------------------------------------------

